package com.mybus17000.ui.screens

import androidx.compose.foundation.layout.*
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.ArrowBack
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.dp
import com.google.android.gms.maps.CameraUpdateFactory
import com.google.android.gms.maps.model.*
import com.google.maps.android.compose.*
import kotlinx.coroutines.launch

@Composable
fun RouteMapScreen(
    fromLatLng: LatLng?,
    toLatLng: LatLng?,
    routePoints: List<LatLng>,
    onBack: () -> Unit
) {
    val cameraState = rememberCameraPositionState()
    val coroutineScope = rememberCoroutineScope()

    LaunchedEffect(routePoints) {
        if (routePoints.isNotEmpty()) {
            val bounds = LatLngBounds.builder().apply {
                routePoints.forEach { include(it) }
            }.build()
            coroutineScope.launch {
                cameraState.animate(
                    update = CameraUpdateFactory.newLatLngBounds(bounds, 150)
                )
            }
        }
    }

    Box(Modifier.fillMaxSize()) {
        // 🗺️ Map
        GoogleMap(
            modifier = Modifier.fillMaxSize(),
            cameraPositionState = cameraState
        ) {
            // 📍 Départ
            fromLatLng?.let {
                Marker(
                    state = MarkerState(position = it),
                    title = "Départ",
                    icon = BitmapDescriptorFactory.defaultMarker(BitmapDescriptorFactory.HUE_GREEN)
                )
            }
            // 📍 Arrivée
            toLatLng?.let {
                Marker(
                    state = MarkerState(position = it),
                    title = "Arrivée",
                    icon = BitmapDescriptorFactory.defaultMarker(BitmapDescriptorFactory.HUE_RED)
                )
            }
            // 🚍 Ligne du parcours
            if (routePoints.isNotEmpty()) {
                Polyline(
                    points = routePoints,
                    color = Color(0xFF1976D2),
                    width = 8f
                )
            }
        }

        // ⬅️ Bouton retour
        IconButton(
            onClick = onBack,
            modifier = Modifier
                .padding(16.dp)
                .align(androidx.compose.ui.Alignment.TopStart)
        ) {
            Icon(
                imageVector = Icons.Default.ArrowBack,
                contentDescription = "Retour",
                tint = Color.White
            )
        }

        // 🧭 Légende / infos
        Card(
            modifier = Modifier
                .align(androidx.compose.ui.Alignment.BottomCenter)
                .padding(16.dp),
            colors = CardDefaults.cardColors(containerColor = Color.White)
        ) {
            Column(Modifier.padding(12.dp)) {
                Text("🟩 Départ — 🟥 Arrivée", style = MaterialTheme.typography.bodyMedium)
                Text("🚍 ${routePoints.size} points de parcours", color = Color.Gray)
            }
        }
    }
}


